; $Id: tia_bank.inc bdupeyron.tech@gmail.com(Antichambre)
;
; MIDIbox SID
; BankStick Handler
;
; (Bank Load/Store routines located in tia_pbank.inc and tia_ebank.inc)
;
; ==========================================================================
;
;  Copyright 1998-2007 Thorsten Klose (tk@midibox.org)
;  Licensed for personal non-commercial use only.
;  All other rights reserved.
; 
; ==========================================================================

;; --------------------------------------------------------------------------
;;  Writes data to EEPROM or BankStick
;;  IN: Value in WREG, address offset in EEADR
;;  OUT: increments address
;; --------------------------------------------------------------------------
TIA_BANK_Write
	movwf	TIA_BANK			; temporary save word in TIA_BANK
	;; branch to internal patch if bankstick is not available or Patch is 0
	rcall	TIA_BANK_GetBankStickAvailable
	bz	TIA_BANK_WriteInternal
	movf	TIA_PRESET, W
	bz	TIA_BANK_WriteInternal

    ;; test BS Size 64/128 Patchs, 
    ;rcall   TIA_BANK_GetBankStickSize
    ;bnz     TIA_BANK_WriteBankStick
    ;; exit if patch exceeds 63
    ;btfsc	TIA_PRESET, 6
    ;rgoto   TIA_BANK_WriteInternal

TIA_BANK_WriteBankStick				; BankStick write:
    movlw   0x00
    addlw   DEFAULT_BS_READONLY     ; exit if BS_READONLY
    bnz  TIA_BANK_WriteBankStick_NOk
TIA_BANK_WriteBankStick_Ok    
	rcall	TIA_BANK_SetBankStickAddress
	movf	MIOS_PARAMETER3, W		; get byte from temp. register
 	call	MIOS_BANKSTICK_Write	; write content
	;; here we could add an error exception handler
TIA_BANK_WriteBankStick_NOk
	incf	EEADR, F                ; increment EEPROM address
	rgoto	TIA_BANK_Write_End

TIA_BANK_WriteInternal
    movlw   0x00
    addlw   DEFAULT_EE_READONLY     ; exit if EE_READONLY
    bz      TIA_BANK_WriteInternal_Ok
	incf	EEADR, F                ; increment EEADR
	rgoto	TIA_BANK_Write_End		; and goto end
	;; don't write if address in range between 0x7c and 0x7f (used to save channel and device number)
	BRA_IFSET EEADR, 7, ACCESS, TIA_BANK_WriteInternal_Ok
	movlw	0x7c-1
	cpfsgt	EEADR, ACCESS
	rgoto   TIA_BANK_WriteInternal_Ok
	incf	EEADR, F                ; increment EEADR
	rgoto	TIA_BANK_Write_End		; and goto end
TIA_BANK_WriteInternal_Ok
	movf	MIOS_PARAMETER3, W		; get byte from temp. register
	call	MIOS_EEPROM_Write		; write to EEPROM
	;; here we could add an error exception handler
TIA_BANK_Write_End
	SET_BSR	TIA_BASE			; fix BSR after a MIOS routine has been called
	return

;; --------------------------------------------------------------------------
;;  Writes a page of 64 bytes to EEPROM or BankStick
;;  IN: Value in FSR1 buffer, address offset in EEADR
;;  OUT: increments address
;; --------------------------------------------------------------------------
TIA_BANK_WritePage

	;; align EEADR if necessary
	movlw	0xc0
	andwf	EEADR, F

	;; branch to internal patch if bankstick is not available or Patch is 0
	movf	TIA_PRESET, W
	bz	TIA_BANK_WritePageInternal

TIA_BANK_WritePageBankStick			; BankStick write:
    movlw   0x00
    addlw   DEFAULT_BS_READONLY     ; exit if BS_READONLY
    bz      TIA_BANK_WritePageBankStick_Ok
TIA_BANK_WritePageBankStick_ReadOnly
    ;;  BS Read Only Error #0x02
    movlw   0x02
    movwf   TIA_SYSEX_ERROR
    rgoto TIA_BANK_WritePageBankStick_End
   
TIA_BANK_WritePageBankStick_Ok    
	rcall	TIA_BANK_SetBankStickAddress
	;; buffer already prepared in FSR1
 	call	MIOS_BANKSTICK_WritePage	; write page
	;; here we could add an error exception handler
    bz      TIA_BANK_WritePageBankStick_End
    ;;  BS WritePage Error #0x07
    movlw   0x07
    movwf   TIA_SYSEX_ERROR
TIA_BANK_WritePageBankStick_End
	movlw	0x40				; increment EEPROM address by 0x40
	addwf	EEADR, F
	rgoto	TIA_BANK_WritePage_End

TIA_BANK_WritePageInternal
    clrf    TIA_PRESET
    movlw   0x00
    addlw   DEFAULT_EE_READONLY     ; exit if EE_READONLY
    bz      TIA_BANK_WritePageInternalLoop
    ;; Internal EE Readonly Error #1
    movlw   0x01
    movwf   TIA_SYSEX_ERROR
    movlw   0x40
	addwf   EEADR, F			; increment EEADR
	rgoto	TIA_BANK_WritePage_End		; and goto next
	;; write 64 bytes
TIA_BANK_WritePageInternalLoop
	;; don't write if address in range between 0x7c and 0x7f (used to save channel and device number)
	BRA_IFSET EEADR, 7, ACCESS, TIA_BANK_WritePageInternal_Ok
	movlw	0x7c-1
	cpfsgt	EEADR, ACCESS
	rgoto   TIA_BANK_WritePageInternal_Ok
	incf	EEADR, F			; increment EEADR
	rgoto	TIA_BANK_WritePage_Next		; and goto next
TIA_BANK_WritePageInternal_Ok
	movf	EEADR, W
	andlw	0x3f
	movf	PLUSW1, W			; get byte from buffer
	call	MIOS_EEPROM_Write		; write to EEPROM
	;; here we could add an error exception handler
TIA_BANK_WritePage_Next
	movf	EEADR, W
	andlw	0x3f
	bnz	TIA_BANK_WritePageInternalLoop

TIA_BANK_WritePage_End
	SET_BSR	TIA_BASE			; fix BSR after a MIOS routine has been called
	return

    
;; --------------------------------------------------------------------------
;;  Read data from EEPROM or BankStick
;;  IN: address offset in EEADR,
;;      internal perset in EEADRH, 
;;      Patch number in MIOS_PARAMETER2,
;;      Bank number in TIA_BANK.
;;  OUT: 7-bit result in WREG, increments address
;; --------------------------------------------------------------------------

TIA_BANK_Read
	;; branch to internal patch if bankstick is not available or Patch is 0    
    movf    TIA_BANK, W
    rcall   TIA_BANK_GetBankStickReady
	bz	TIA_BANK_ReadInternal
	movf	TIA_PRESET, W
	bz	TIA_BANK_ReadInternal
     
    rcall   TIA_BANK_GetBankStickSize
    bnz     TIA_BANK_ReadBankStick

    btfsc	TIA_PRESET, 6
    rgoto   TIA_BANK_Read_End    

TIA_BANK_ReadBankStick				; BankStick read:
	rcall	TIA_BANK_SetBankStickAddress
	call	MIOS_BANKSTICK_Read		; read content
	incf	EEADR, F			; increment EEPROM address
	rgoto	TIA_BANK_Read_End
	
TIA_BANK_ReadInternal
	clrf	TIA_PRESET
	call	MIOS_EEPROM_Read

TIA_BANK_Read_End
	SET_BSR	TIA_BASE			; fix BSR after a MIOS routine has been called
	return


;; --------------------------------------------------------------------------
;;  This function sets a BankStick address
;;  IN: patch offset in EEADR
;;      patch number in MIOS_PARAMETER2
;;      BankStick number in TIA_BANK
;;  OUT: address in MIOS_PARAMETER[12]
;;       BankStick selected via MIOS_BANKSTICK_CtrlSet
;; --------------------------------------------------------------------------
TIA_BANK_SetBankStickAddressMagic
    clrf    MIOS_PARAMETER1
    clrf    MIOS_PARAMETER2
    rgoto TIA_BANK_SetBankStickAddress_Cont

TIA_BANK_SetBankStickAddress

	movff	EEADR, MIOS_PARAMETER1		; copy address to low-byte
    btfsc	TIA_PRESET, 0
	bsf     MIOS_PARAMETER1, 7		; select upper address range on odd bank number
    rrncf   TIA_PRESET, W      ; high-byte
    movwf   MIOS_PARAMETER2

TIA_BANK_SetBankStickAddress_Cont
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_BANKSTICK_CtrlSet
        
    rcall   TIA_BANK_GetBankStickSize
	skpnz
    rgoto   TIA_BANK_SetBankStickAddress_32k
    
TIA_BANK_SetBankStickAddress_64k
	btfsc	TIA_BANK, 0
	bsf	MIOS_PARAMETER2, 6
    btfsc	TIA_BANK, 1
	bsf	MIOS_PARAMETER2, 7
    rgoto   TIA_BANK_SetBankStickAddress_End
    
TIA_BANK_SetBankStickAddress_32k
    bcf	MIOS_PARAMETER2, 5
	btfsc	TIA_BANK, 0
	bsf	MIOS_PARAMETER2, 5
    btfsc	TIA_BANK, 1
	bsf	MIOS_PARAMETER2, 6

TIA_BANK_SetBankStickAddress_End
    return
    

;; --------------------------------------------------------------------------
;;  This function returns the bankstick available status for the current bank
;;  IN: Bank number WREG
;;  OUT: WREG != 0 when bankstick available
;; --------------------------------------------------------------------------
TIA_BANK_GetBankStickReady
    rrncf   WREG, W
    rrncf   WREG, W
    andlw   0x07
    movwf	TIA_BANKSTICK_ID
	call	MIOS_HLP_GetBitORMask
	andwf	TIA_BANKSTICK_RDY, W
	return
    
;; --------------------------------------------------------------------------
;;  This function returns the bankstick available status for the current bank
;;  IN: BankStick number in TIA_PBANK
;;  OUT: WREG != 0 when bankstick available
;; --------------------------------------------------------------------------
TIA_BANK_GetBankStickAvailable
    movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	andwf	TIA_BANKSTICK_STAT, W
	return
;; --------------------------------------------------------------------------
;;  This function returns the bankstick available status for the current bank
;;  IN: BankStick number in TIA_BANKSTICK_ID
;;  OUT: BankStick size in WREG and TIA_BANKSTICK_SIZE
;; --------------------------------------------------------------------------
TIA_BANK_GetBankStickSize
    movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	andwf	TIA_BANKSTICK_SIZE, W
	return
    


;; --------------------------------------------------------------------------
;; selects address of magic number depending on TIA_BANKSTICK_CHK_CTR
;; TIA_BANKSTICK_CHK_CTR or TIA_PBANK
;; --------------------------------------------------------------------------
TIA_BANK_CheckStick_SelectMagic
	;; select BankStick depending on TIA_BANKSTICK_CHK_CTR
    movwf   MIOS_PARAMETER3
    rrncf   WREG, W
    rrncf   WREG, W
    andlw   0x07
    addlw   0x80	; (Enable verify slower write accesses)
	call	MIOS_BANKSTICK_CtrlSet

	;; determine if this is a 64k BankStick:
	;; read from address 0x8000, store value in PRODL
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	movlw	0x80
	movwf	MIOS_PARAMETER2
	call	MIOS_BANKSTICK_Read
	movwf	PRODL

	;; add 0x42 and write number to 0x0000
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	clrf	MIOS_PARAMETER2
    movf    PRODL, W
	addlw	0x42
	call	MIOS_BANKSTICK_Write

	;; read again number from 0x8000, check if we still see the old value
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	movlw	0x80
	movwf	MIOS_PARAMETER2
	call	MIOS_BANKSTICK_Read
	xorwf	PRODL, W
    bnz     TIA_BANK_CheckStick_SelectMagic_32k
TIA_BANK_CheckStick_SelectMagic_64k	; (64k device)
	;; try to read from BankStick (address 0x0000, 0x4000, 0x8000 or 0xc000)
	clrf	MIOS_PARAMETER1
	clrf	MIOS_PARAMETER2
	btfsc	MIOS_PARAMETER3, 0
	bsf	MIOS_PARAMETER2, 6
    btfsc	MIOS_PARAMETER3, 1
	bsf	MIOS_PARAMETER2, 7
    rgoto   TIA_BANK_CheckStick_SelectMagic_End
    
TIA_BANK_CheckStick_SelectMagic_32k	; (32k device)
	;; try to read from BankStick (address 0x0000, 0x2000, 0x4000 or 0x0600)
	clrf	MIOS_PARAMETER1
	clrf	MIOS_PARAMETER2
	btfsc	MIOS_PARAMETER3, 0
	bsf	MIOS_PARAMETER2, 5
    btfsc	MIOS_PARAMETER3, 1
	bsf	MIOS_PARAMETER2, 6
    ;; rgoto   TIA_BANK_CheckStick_SelectMagic_End
TIA_BANK_CheckStick_SelectMagic_End
	return





;; --------------------------------------------------------------------------
;;  Check Stick: try to read from BankStick, clear the appr. flag in
;;  TIA_BANKSTICK_STAT if BankStick not available
;; --------------------------------------------------------------------------
TIA_BANK_CheckStick

	;; increment check counter, wrap at 32
	incf	TIA_BANKSTICK_CHK_CTR, W
	andlw	0x1f
	movwf	TIA_BANKSTICK_CHK_CTR
    
    clrc
	rrncf   TIA_BANKSTICK_CHK_CTR, W
    rrncf   WREG, W
    andlw   0x07
    movwf   TIA_BANKSTICK_ID
    
	;; select "magic number" and try to read from BankStick
    movf    TIA_BANKSTICK_CHK_CTR, W
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read
	;; this sets the MIOS_BOX_STAT_BS_AVAILABLE flag

	;; save old BankStick status in TMP1
	movff	TIA_BANKSTICK_STAT, TMP1

	;; modify the available flag of current bankstick
	BRA_IFSET MIOS_BOX_STAT, MIOS_BOX_STAT_BS_AVAILABLE, ACCESS, TIA_BANK_CheckStick_AccPassed
TIA_BANK_CheckStick_AccFailed
	movf    TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_STAT, W
	rgoto	TIA_BANK_CheckStick_AccCont
TIA_BANK_CheckStick_AccPassed
	movf    TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	iorwf	TIA_BANKSTICK_STAT, W
TIA_BANK_CheckStick_AccCont
    movwf   TMP2

	;; check if flag has been changed
	movf	TMP2, W
	xorwf	TMP1, W
    bz  TIA_BANK_CheckStick_End
    bsf     TIA_STAT, TIA_STAT_ENGINE_DISABLE, ACCESS

    ;; flag changed - branch depending on available status
	movf    TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	andwf	TMP2, W
	bnz	TIA_BANK_CheckStick_Ext
    rgoto TIA_BANK_CheckStick_Int
    bcf     TIA_STAT, TIA_STAT_ENGINE_DISABLE
TIA_BANK_CheckStick_End    
    return




	;; ------------------------------------------------------------------
	;; --> Internal Patch
TIA_BANK_CheckStick_Int

    ;; reset size
    movf	TMP2, W
	andwf	TIA_BANKSTICK_SIZE, F

TIA_BANK_CheckStick_Int_StatOk
    movff   TMP2, TIA_BANKSTICK_STAT
    call    TIA_LEDMTR_PlayOtoR         ;; play mled orange to red pattern 
    call	TIA_TUNE_PlayDisconnect     ;; play tune 5    
    call    TIA_LEDMTR_Player_Off
    
	;; branch depending on patch/kit BankStick
#if DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7
	movlw	DEFAULT_BS_KBANK_ID-1
	cpfsgt	TIA_BANKSTICK_ID, ACCESS
	rgoto	TIA_BANK_CheckStick_Int_P
    btfsc   TIA_BANKSTICK_CHK_CTR, 1
    rgoto	TIA_BANK_CheckStick_Int_WT
    
TIA_BANK_CheckStick_Int_K
	movf	TIA_BANKSTICK_ID, W         ;; clear flag
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_RDY, F   
    movlw   0x05 ;;send CFG_BSInfo      ;; send sysex CFG BS infos
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read 
    movf    TIA_BANKSTICK_CHK_CTR, W
    xorwf   TIA_KBANK, W
    bnz     TIA_BANK_CheckStick_Int_End
    clrf    TIA_KBANK                   ;; Reinit to...
    clrf    TIA_KIT
    movlw   0x0f                        ;; send sysex CFG All
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read     ;; ...internal patch
	rgoto	TIA_BANK_CheckStick_Int_End 
    
TIA_BANK_CheckStick_Int_WT
	movf	TIA_BANKSTICK_ID, W         ;; clear flag
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_RDY, F   
    movlw   0x05 ;;send CFG_BSInfo      ;; send sysex CFG BS infos
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read 
    movf    TIA_BANKSTICK_CHK_CTR, W
    xorwf   TIA_WBANK, W
    bnz     TIA_BANK_CheckStick_Int_End
    clrf    TIA_WBANK                   ;; Reinit to...
    clrf    TIA_WT
    movlw   0x0f                        ;; send sysex CFG All
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read     ;; ...internal patch
	rgoto	TIA_BANK_CheckStick_Int_End 
#endif  ;; DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7

TIA_BANK_CheckStick_Int_P
	movf	TIA_BANKSTICK_ID, W         ;; clear flag
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_RDY, F   
    movlw   0x05 ;;send CFG_BSInfo      ;; send sysex CFG BS infos
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read 
    movf    TIA_BANKSTICK_CHK_CTR, W
    xorwf   TIA_PBANK, W
    bnz     TIA_BANK_CheckStick_Int_End
    clrf    TIA_PBANK                   ;; Reinit to...
    clrf    TIA_PATCH
    movlw   0x0f                        ;; send sysex CFG All
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read     ;; ...internal patch
	call	TIA_PATCH_Init
	;;rgoto	TIA_BANK_CheckStick_Int_End 
    
TIA_BANK_CheckStick_Int_End    
	return


	;; ------------------------------------------------------------------
	;; --> External Patch
TIA_BANK_CheckStick_Ext
	;; 	TABLE_ADDR TEXT_EXTBANK_0		; print message
	;; 	call	MIOS_LCD_PrintMessage
	;; 	call	MIOS_LCD_PrintMessage

	;; new setup will be reloaded automatically
	;; set reinit counter - CS configuration will be restored after one second
	;movlw	10
	;movwf	CS_MENU_REINIT_CFG_CTR

	;; now check if the magic numbers exist in bankstick bank - if not, format bank automatically
	movlw	50				; wait some ms to get a stable status
	call	MIOS_Delay			

    movf    TIA_BANKSTICK_CHK_CTR, W
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read
	xorlw	BANKSTICK_MAGIC0		; branch to unformatted message if number not equal
    skpz
	rgoto	TIA_BANK_CheckStick_Unformatted
    
    ;; magic numbers are different between Patch/Kit-WT bankstick
#if DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7
	movlw	DEFAULT_BS_KBANK_ID-1
	cpfsgt	TIA_BANKSTICK_ID, ACCESS
	rgoto	TIA_BANK_CheckStick_Ext_P

;; Kit / Wavetable
TIA_BANK_CheckStick_Ext_K 
    ;; Magic ( Kit suite)
    ;; set bankstick type depending on second byte (BANKSTICK_MAGICK/P)
	call	MIOS_BANKSTICK_Read 
    movwf   TMP1
	xorlw	BANKSTICK_MAGICK
	bz	TIA_BANK_CheckStrick_Ext_Type_K
TIA_BANK_CheckStrick_Ext_Type_NotK
    movf    TMP1, W
    xorlw	BANKSTICK_MAGICP
    skpz
    rgoto	TIA_BANK_CheckStick_Unformatted             ;; format
    movlw   DEFAULT_BS_FPROTECT          ;; type format protection
    xorlw   0x01
    skpnz
    rgoto   TIA_BANK_CheckStick_Unformatted_TProtect    ;; exit
    rgoto   TIA_BANK_CheckStick_Unformatted             ;; format
TIA_BANK_CheckStrick_Ext_Type_K
    movlw   BANKSTICK_MAGICK            ;; valid type
    movwf   TMP1
	rgoto	TIA_BANK_CheckStick_Ext_Size
#endif      ;; DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7

;; Patch
TIA_BANK_CheckStick_Ext_P 
    ;; Magic (Patch suite)
    ;; set bankstick type depending on second byte (BANKSTICK_MAGICK/P)
	call	MIOS_BANKSTICK_Read 
    movwf   TMP1
	xorlw	BANKSTICK_MAGICP
	bz	TIA_BANK_CheckStrick_Ext_Type_P
TIA_BANK_CheckStrick_Ext_Type_NotP
    movf    TMP1, W
    xorlw	BANKSTICK_MAGICK            
    skpz
    rgoto	TIA_BANK_CheckStick_Unformatted             ;; format
    movlw   DEFAULT_BS_FPROTECT          ;; type format protection
    xorlw   0x01
    skpnz
    rgoto   TIA_BANK_CheckStick_Unformatted_TProtect    ;; exit
    rgoto   TIA_BANK_CheckStick_Unformatted             ;; format
TIA_BANK_CheckStrick_Ext_Type_P         ;; valid type
    movlw   BANKSTICK_MAGICP
    movwf   TMP1
	;;rgoto	TIA_BANK_CheckStick_Ext_Size

TIA_BANK_CheckStick_Ext_Size
    ;; set bankstick size depending on third byte (32=32k, 64=64k)
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_SIZE, F       ;; clear size
	call	MIOS_BANKSTICK_Read
	xorlw	64
	bnz	TIA_BANK_CheckStrick_Ext_Not64k
TIA_BANK_CheckStrick_Ext_64k
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	iorwf	TIA_BANKSTICK_SIZE, F       ;; set size
TIA_BANK_CheckStrick_Ext_Not64k

TIA_BANK_CheckStick_Ext_Fix ; to do if necessary...
	;; fix patches if required (and enabled via ENABLE_PATCH_FIXING)
    ;movff	TIA_BANKSTICK_CHK_CTR, TIA_PBANK
	;rcall	TIA_BANK_FixPatches
	;; ok
    
TIA_BANK_CheckStick_Ext_Stat

    movf    TIA_BANKSTICK_CHK_CTR, W
    
    movf    TIA_BANKSTICK_CHK_CTR, W    ;; test 1st bank of current BS
    andlw   0x1c
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read         ;; read first byte from BankStick (magic0)
	xorlw	BANKSTICK_MAGIC0            ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
	call	MIOS_BANKSTICK_Read         ;; read second byte from BankStick (magicPK)
	xorwf	TMP1, W                     ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk

    movf    TIA_BANKSTICK_CHK_CTR, W    ;; test 2nd bank of current BS
    andlw   0x1c
    addlw   0x01
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read         ;; read first byte from BankStick (magic0)
	xorlw	BANKSTICK_MAGIC0            ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
	call	MIOS_BANKSTICK_Read         ;; read second byte from BankStick (magicPK)
	xorwf	TMP1, W                     ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
    
    movf    TIA_BANKSTICK_CHK_CTR, W    ;; test 3rd bank of current BS
    andlw   0x1c
    addlw   0x02
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read         ;; read first byte from BankStick (magic0)
	xorlw	BANKSTICK_MAGIC0            ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
	call	MIOS_BANKSTICK_Read         ;; read second byte from BankStick (magicPK)
	xorwf	TMP1, W                     ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
    
    movf    TIA_BANKSTICK_CHK_CTR, W    ;; test 4th bank of current BS
    andlw   0x1c
    addlw   0x03
	rcall	TIA_BANK_CheckStick_SelectMagic
	call	MIOS_BANKSTICK_Read         ;; read first byte from BankStick (magic0)
	xorlw	BANKSTICK_MAGIC0            ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk
	call	MIOS_BANKSTICK_Read         ;; read second byte from BankStick (magicPK)
	xorwf	TMP1, W                     ;; branch to unformatted message if number not equal
	bnz	TIA_BANK_CheckStick_Ext_Stat_NotOk

TIA_BANK_CheckStick_Ext_Stat_Ok   
    movff   TMP2, TIA_BANKSTICK_STAT    ;; status is set
    rgoto TIA_BANK_CheckStick_Ext_ReadySet
TIA_BANK_CheckStick_Ext_Stat_NotOk      ;; status not ready
    rgoto   TIA_BANK_CheckStick_Ext_End ;; end
    
TIA_BANK_CheckStick_Unformatted    
    movlw   DEFAULT_BS_FPROTECT          ;; BS format inhibit
    xorlw   0x02
    bz      TIA_BANK_CheckStick_Unformatted_FProtect 

    movff   TIA_BANKSTICK_CHK_CTR, TIA_BANK
	rcall	TIA_BANK_FormatStick		;; --> format bankstick
    movff   TIA_BANK, TIA_BANKSTICK_CHK_CTR
    rgoto	TIA_BANK_CheckStick_Ext_End ;; end
    
TIA_BANK_CheckStick_Unformatted_FProtect
    movff   TMP2, TIA_BANKSTICK_STAT    ;; status is set
    rgoto	TIA_BANK_CheckStick_Ext_ReadyClr

TIA_BANK_CheckStick_Unformatted_TProtect
    movff   TMP2, TIA_BANKSTICK_STAT    ;; status is set
    rgoto	TIA_BANK_CheckStick_Ext_ReadyClr

TIA_BANK_CheckStick_Ext_ReadyClr
	movf	TIA_BANKSTICK_ID, W         ;; clear flag
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_RDY, F   
    call    TIA_LEDMTR_PlayOtoR         ;; play mled orange to red pattern 
    call	TIA_TUNE_PlayWrong          ;; play wrong tune     
    call    TIA_LEDMTR_Player_Off
    movlw   0x05 ;;send CFG_BSInfo      ;; send sysex CFG BS infos
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read 
    movf    TIA_BANKSTICK_CHK_CTR, W
    xorwf   TIA_PBANK, W
    bnz     TIA_BANK_CheckStick_Ext_End
    clrf    TIA_PBANK                   ;; Reinit to...
    clrf    TIA_PRESET
    movlw   0x0f                        ;; send sysex CFG All
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read     ;; ...internal patch
    bcf     TIA_STAT, TIA_STAT_ENGINE_DISABLE
	goto	TIA_PATCH_Init
    ;;rgoto   TIA_BANK_CheckStick_Ext_End 
    
TIA_BANK_CheckStick_Ext_ReadySet 
	movf	TIA_BANKSTICK_ID, W         ;; set flag
	call	MIOS_HLP_GetBitORMask
	iorwf	TIA_BANKSTICK_RDY, F
    call    TIA_LEDMTR_PlayOtoG         ;; play mled orange to green pattern 
    call	TIA_TUNE_PlayOk             ;; play ready tune    
    call    TIA_LEDMTR_Player_Off
    movlw   0x05 ;;send CFG_BSInfo      ;; send sysex CFG BS infos
    movwf   TIA_SYSEX_ADDRESS
	call    _TIA_SYSEX_End_CFG_Read 
    bcf     TIA_STAT, TIA_STAT_ENGINE_DISABLE
    ;;rgoto   TIA_BANK_CheckStick_Ext_End

TIA_BANK_CheckStick_Ext_End
	return
       
    
    
;; --------------------------------------------------------------------------
;;  Format Stick: copy the lead default patch into the currently selected BankStick Bank
;;  IN: bank number in TIA_PBANK
;; --------------------------------------------------------------------------
TEXT_FORMATBANK_0 STRING 20, 0x00, "* Formatting xxk *  "
TEXT_FORMATBANK_1 STRING 20, 0x40, "* Patch x  0     *  "
TEXT_FORMATBANK_K STRING 20, 0x40, "* Ensemble 000   *  "

TIA_BANK_FormatStick
	;; select BankStick depending on TIA_BANKSTICK_CHK_CTR
    rrncf   TIA_BANK, W
    rrncf   WREG, W
    andlw   0x07
    movwf	TIA_BANKSTICK_ID
	call	MIOS_BANKSTICK_CtrlSet
    
    call    TIA_LEDMTR_PlayFormat_R

	;; determine if this is a 64k BankStick:
	;; read from address 0x8000, store value in PRODL
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	movlw	0x80
	movwf	MIOS_PARAMETER2
	call	MIOS_BANKSTICK_Read
	movwf	PRODL

	;; add 0x42 and write number to 0x0000
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	clrf	MIOS_PARAMETER2
    movf    PRODL, W
	addlw	0x42
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler

	;; read again number from 0x8000, check if we still see the old value
	movlw	0x0f ;; between 0x03 - 0x7f
	movwf	MIOS_PARAMETER1
	movlw	0x80
	movwf	MIOS_PARAMETER2
	call	MIOS_BANKSTICK_Read
	xorwf	PRODL, W
    bnz     TIA_BANK_FormatStick_32k
TIA_BANK_FormatStick_64k
	;; set flag in size register
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	iorwf	TIA_BANKSTICK_SIZE, F
	;; write 64 to address 0x0002
	movlw	64
	rgoto	TIA_BANK_FormatStick_SizeIDCont
TIA_BANK_FormatStick_32k
	;; clear flag in size register
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_SIZE, F
	;; write 32 to address 0x0002
	movlw	32
	;;rgoto	TIA_BANK_FormatStick_SizeIDCont
TIA_BANK_FormatStick_SizeIDCont
	movwf	TMP1
    movf    TIA_BANKSTICK_CHK_CTR, W
	rcall	TIA_BANK_CheckStick_SelectMagic
	movlw	0x02
	movwf	MIOS_PARAMETER1
	movf	TMP1, W
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler

	movlw	50				; wait some ms to get a stable status
	call	MIOS_Delay			

	;; now branch depending on Ensemble/Patch
#if DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7
	movlw	DEFAULT_BS_KBANK_ID-1
	cpfsgt	TIA_BANKSTICK_ID, ACCESS
	rgoto	TIA_BANK_FormatStick_P
    btfsc   TIA_BANK, 1
    rgoto	TIA_BANK_FormatStick_WT
    
TIA_BANK_FormatStick_K
	;; copy Patch 0 (internal kit) to upload buffer
	clrf	EEADR
    movlw   (EEPROM_KIT >> 8) & 0xff
    movwf   EEADRH
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_K_CIntLoop
	call	MIOS_EEPROM_Read
	movwf	POSTINC0
	btfss	EEADR, 7
	rgoto	TIA_BANK_FormatStick_K_CIntLoop
	;; clear name
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_K_ClearLoop
	clrf	POSTINC0
	BRA_IFCLR FSR0L, 4, ACCESS, TIA_BANK_FormatStick_K_ClearLoop
    rgoto   TIA_BANK_FormatStick_Data    
    
TIA_BANK_FormatStick_WT
	;; copy Patch 0 (internal patch) to upload buffer
	clrf	EEADR
    movlw   (EEPROM_WAVETABLE >> 8) & 0xff
    movwf   EEADRH
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_WT_CIntLoop
	call	MIOS_EEPROM_Read
	movwf	POSTINC0
	btfss	EEADR, 7
	rgoto	TIA_BANK_FormatStick_WT_CIntLoop
	;; clear name
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_WT_ClearLoop
	clrf	POSTINC0
	BRA_IFCLR FSR0L, 4, ACCESS, TIA_BANK_FormatStick_WT_ClearLoop
    rgoto   TIA_BANK_FormatStick_Data

#endif  ;; DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7

TIA_BANK_FormatStick_P
	;; copy Patch 0 (internal patch) to upload buffer
	clrf	EEADR
    movlw   (EEPROM_PATCH >> 8) & 0xff
    movwf   EEADRH
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_P_CIntLoop
	call	MIOS_EEPROM_Read
	movwf	POSTINC0
	btfss	EEADR, 7
	rgoto	TIA_BANK_FormatStick_P_CIntLoop
	;; clear name
	lfsr	FSR0, BANKSTICK_FORMAT_BEGIN
TIA_BANK_FormatStick_P_ClearLoop
	clrf	POSTINC0
	BRA_IFCLR FSR0L, 4, ACCESS, TIA_BANK_FormatStick_P_ClearLoop
    ;;rgoto   TIA_BANK_FormatStick_Data



    
TIA_BANK_FormatStick_Data  
    call    TIA_LEDMTR_PlayFormat_O
	;; now format patch 1 to 127 (64k) or 63 (32k)
	movlw	0x01
	movwf	EEADR	; use EEADR as patch ctr
    ;; Select BS Adress
    clrf    MIOS_PARAMETER1
    btfsc	EEADR, 0
	bsf     MIOS_PARAMETER1, 7		; select upper address range on odd bank number
    rrf     EEADR, W 	
	movwf	MIOS_PARAMETER2	; copy patch to high-byte
    rcall   TIA_BANK_GetBankStickSize
	skpnz
    rgoto   TIA_BANK_FormatStick_Data_32k
TIA_BANK_FormatStick_Data_64k
	btfsc	TIA_BANK, 0
	bsf	MIOS_PARAMETER2, 6
    btfsc	TIA_BANK, 1
	bsf	MIOS_PARAMETER2, 7
    rgoto   TIA_BANK_FormatStick_Data_OuterLoop
TIA_BANK_FormatStick_Data_32k
    bcf	MIOS_PARAMETER2, 5
	btfsc	TIA_BANK, 0
	bsf	MIOS_PARAMETER2, 5
    btfsc	TIA_BANK, 1
	bsf	MIOS_PARAMETER2, 6
    
TIA_BANK_FormatStick_Data_OuterLoop
	;; copy 0x80 words
	lfsr	FSR1, BANKSTICK_FORMAT_BEGIN 
	clrwdt			; feed watchdog
	;; buffer already prepared in FSR1
 	call	MIOS_BANKSTICK_WritePage	; write page
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
	movlw	0x40				; increment FSR1 by 0x40
	addwf	FSR1L, F
	clrwdt			; feed watchdog
 	call	MIOS_BANKSTICK_WritePage	; write page
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
	;; continue until last patch is reached
	incf	EEADR, F
	movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitORMask
	andwf	TIA_BANKSTICK_SIZE, W
	movlw	128-1
	skpnz
	movlw	64-1
	cpfsgt	EEADR, ACCESS
	rgoto TIA_BANK_FormatStick_Data_OuterLoop
TIA_BANK_FormatStick_Data_End
     
    call    TIA_LEDMTR_PlayFormat_G1
    
    ;; write bank new name
    ;; add name offset to MP1 start @0x10
    movlw   NEW_BANK_NAME & 0xff
	movwf	EEADR
    movlw   (NEW_BANK_NAME >> 8) & 0xff
    movwf   EEADRH 
    movf    TIA_BANKSTICK_CHK_CTR, W
	rcall	TIA_BANK_CheckStick_SelectMagic
    movlw   0x10
    movwf   MIOS_PARAMETER1
TIA_BANK_FormatStick_Name_Loop
    call    MIOS_EEPROM_Read
    call    MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
    btfss   MIOS_PARAMETER1, 5    ; until == 0x20
    rgoto   TIA_BANK_FormatStick_Name_Loop
       
	;;; write magic byte 0 to confirm valid content
    clrf    MIOS_PARAMETER1
	movlw	BANKSTICK_MAGIC0
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
    
#if DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7
    ;; now branch depending on Kit/Patch
	movlw	DEFAULT_BS_KBANK_ID-1
	cpfsgt	TIA_BANKSTICK_ID, ACCESS
	rgoto	TIA_BANK_FormatStick_P_Cont
    btfsc   TIA_BANK, 1
    rgoto	TIA_BANK_FormatStick_WT_Cont
    
TIA_BANK_FormatStick_K_Cont
	;; select first external kit
	;movlw	1
	;movwf	TIA_KIT
	;; store number in EEPROM
	;call	TIA_KIT_StoreDefaultNum
	;; write magic byte K(kit/wavetable), valid content and type
	movlw	BANKSTICK_MAGICK
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
    rgoto   TIA_BANK_FormatStick_Cont

TIA_BANK_FormatStick_WT_Cont 
	;; select first external wavetable
	;movlw	1
	;movwf	TIA_WT
	;; store number in EEPROM
	;call	TIA_KIT_StoreDefaultNum
	;; write magic byte K(kit/wavetable), valid content and type
	movlw	BANKSTICK_MAGICK
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
    rgoto   TIA_BANK_FormatStick_Cont
#endif  ;; DEFAULT_BS_KBANK_ID >= 0 && DEFAULT_BS_KBANK_ID <= 7

TIA_BANK_FormatStick_P_Cont
	;; select first external patch
	;movlw	1
	;movwf	TIA_PRESET
	;; store number in EEPROM
	;call	TIA_KIT_StoreDefaultNum
	;; and branch to end
    ;;rgoto    TIA_BANK_FormatStick_Cont
	;; write magic byte P(patch), valid content and type
	movlw	BANKSTICK_MAGICP
	call	MIOS_BANKSTICK_Write
    skpz
    rgoto   TIA_BANK_FormatStick_Err_Handler
    ;;rgoto   TIA_BANK_FormatStick_Cont

TIA_BANK_FormatStick_Cont
	;; play tune, new setup will be reloaded automatically
	;movf	TIA_PBANK, W
	call	TIA_TUNE_Play1
    call    TIA_LEDMTR_PlayFormat_G2
    rgoto   TIA_BANK_CheckStick_End 

    
TIA_BANK_FormatStick_Err_Handler 
    ;; realign on first bank of current BankStick
    movf    TIA_BANK, F
    andlw   0x1c
    incf    WREG, W    ;; next will be first
    movwf   TIA_BANK   
    ;; reinit STAT registers
    movf	TIA_BANKSTICK_ID, W
	call	MIOS_HLP_GetBitANDMask
	andwf	TIA_BANKSTICK_STAT, F
    andwf	TIA_BANKSTICK_SIZE, F

TIA_BANK_FormatStickEnd
	return

;; --------------------------------------------------------------------------

