/* 
 * =============================================================================
 * 															MIDIBOX PROJECT
 * =============================================================================
 */

//---------------------- General------------------------------------------------

#define OFF    0
#define ON     1

#define LIN    0        // Fonctionnement infra-rouge linaire
#define LOG    1        // Fonctionnement infra-rouge logarithmique (natif)

//---------------------- Modes & Gestion interne -------------------------------

//>>> General <<<

#define 			MODE_VOICE  0
#define			 	MODE_EDIT		1
#define				MODE_EDIT2	2
#define 			MODE_STORE	3
#define				MODE_BANK		4

unsigned char MEM_BUSY;

//>>> Edition avance (Softpots) <<<

#define				SP_MODE_CC				0
#define				SP_MODE_NOTE			1
#define				SP_RELEASE_RTZ		0
#define				SP_RELEASE_KEEP		1

unsigned char ADK_MODE;	     		// MODE_PERF / MODE_EDIT / MODE_STORE
unsigned char GLOBAL_LAST_TOUCHED_PIN;	// Pin du controleur qui a t touch
unsigned char GLOBAL_LAST_TOUCHED_IDX; // Index du controleur qui a t touch
unsigned char EDIT_POS;					// Position du curseur pour l'dition du nom

unsigned char GLOBAL_EDIT_ADV_PIN; // Pin du controleur en edition avance
unsigned char GLOBAL_EDIT_ADV_IDX; // Index du controleur en edition avance

unsigned char SPECIAL_CALL;
unsigned char BTN_FREEZE_PUSHED;

// --------------------- LCD & Encoders ----------------------------------------

#define POS_LINE_1		0x00
#define POS_LINE_2		0x40
#define POS_LINE_3		0x80
#define POS_LINE_4		0xC0

#define DATAWHEEL_SMALL 0
#define DATAWHEEL_BIG   1

//---------------------- Paramtres des controleurs-----------------------------

/*
 Ordre d'index des controleurs :
 0:J1X / 1:J1Y / 2:J2X / 3:J2Y / 4:J3X / 5:J3Y / 6:J4X / 7:J4Y
 8:PITCH / 9:MOD / 10:DBEAM1 / 11:DBEAM2 / 12:POT_LEFT / 13:POT_RIGHT / 14:SOFTPOT1 / 15:SOFTPOT2
*/
/*
const char DEF_NAME_PART1[17][9] = 
{
 "Joystick",
 "Joystick",
 "Joystick",
 "Joystick",
 "Joystick",
 "Joystick",
 "Joystick",
 "Joystick",
 "Pitch Be",
 "Modulati",
 "DBeam Le",
 "DBeam Ri",
 "Basic Kn",
 "Basic Kn",
 "Ribbon C",
 "Ribbon C",
 "Expressi"
};

const char DEF_NAME_PART2[17][9] = 
{
	" 1 X    ",
	" 1 Y    ",
	" 2 X    ",
	" 2 Y    ",
	" 3 X    ",
	" 3 Y    ",
	" 4 X    ",
	" 4 Y    ",
	"nd Wheel",
	"on Wheel",
	"ft      ",
	"ght     ",
	"ob 1    ",
	"ob 2    ",
	"trl V   ",
	"trl H   ",
	"on Pedal"
};
*/
// Paramtres des controleurs (initialiss dans la fonction Init() du main.c)
unsigned char CTL_NAME_PART1[17][9];
unsigned char CTL_NAME_PART2[17][9];

//unsigned char CTL_NAME2[16][16];
unsigned char CTL_LAST_VALUE[17];
unsigned char CTL_MIN[17];
unsigned char CTL_MAX[17];

//unsigned int	sMin[17];			// 10-bit minimum sense (ignore below; => 7bit:   0)
//unsigned int	sMax[17];			// 10-bit maximum sense (ignore above; => 7bit: 127)
//unsigned char sFactor[17];	// calibrated sensor factor: (sMax - sMin) / 127

unsigned char CTL_CC[17];
unsigned char CTL_MIDICH[17];
unsigned char CTL_INV[17];

// Paramtres avancs spcifiques aux DBeams :
unsigned char DB_LINLOG[2];

// Paramtres avancs spcifiques aux SoftPots :
unsigned char SP_MODE[2];   // Mode : CC / Note (Emission d'un controleur continu ou bien d'une note)
unsigned char SP_RELEASE[2];// Release : Return to zero (RTZ) / Keep
														//		> En mode CC   : mode natif
														//		> En mode Note : Note off sur relchement / Latch

// Stockage de 0 et de 1 pour chaque controleur
// Ordre : DBeam1/DBeam2/SoftPot1/SoftPot2
			

unsigned char DWSMALL;
unsigned char DWBIG;

// --------------------- Dfinition des pins AIN -------------------------------

#define J2X_PIN  		 6  // Joystick analogique 2 axe X    *
#define J2Y_PIN  		 4  // Joystick analogique 2 axe Y    *

#define J1X_PIN  		 2  // Joystick analogique 1 axe X    *
#define J1Y_PIN  		 0  // Joystick analogique 1 axe Y    *

#define J4X_PIN  		 7  // Joystick PS2 axe X             *
#define J4Y_PIN  		 5  // Joystick PS2 axe Y             *
#define J3X_PIN  		 1  // Joystick PS2 axe X             *
#define J3Y_PIN  		 3  // Joystick PS2 axe Y             *

#define EXPPED_PIN    9 // Expression pedal               *
#define DBEAM1_PIN	 10 // GP2D12 GAUCHE                  *
#define DBEAM2_PIN	  8 // GP2D12 DROITE
#define SOFTPOT1_PIN 12 // Ribbon Controller (short)      *
#define SOFTPOT2_PIN 14 // Ribbon Controller (long)       *

#define PITCH_PIN    16 //24 // Pitch bend wheel
#define MOD_PIN      18 //26 // Modulation wheel
#define POT_LEFT     20 //28 // Potard
#define POT_RIGHT    22 //30 // Potard

#define NB_AIN       23 // Utiliss : 0-22 = 23 entres AIN
                        // (sauf 11,13,15 sans capteur)

// --------------------- Dfinition des pins DIN -------------------------------

#define BTN_VOICE    1
#define BTN_INTCRT   2
#define BTN_BANK     3
#define BTN_EDIT     5
#define BTN_STORE    6
#define BTN_PERF     7
#define BTN_DWSMALL	 9
#define BTN_DEC     12
#define BTN_ARROWL  13
#define BTN_ARROWR  14
#define BTN_INC     15
#define BTN_DWBIG   17

#define BTN_FREEZE  20 
#define BTN_RIBBS   21 // Ribbons ON/OFF
#define BTN_BEAM    22
#define BTN_CLONE   23
#define BTN_TAP     25
#define BTN_INV     26
#define BTN_DBONOFF 27

#define BTN_PEDUP   30 // Switch pedal up
#define BTN_PEDDOWN 31 // Switch pedal down

// --------------------- Dfinition des pins DOUT ------------------------------

// LEDS

#define LED_ARROWR 0 
#define LED_ARROWL 1
#define LED_INTCRT 2
#define LED_PERF   3
#define LED_BANK   4
#define LED_VOICE  5

#define LED_PS2    8
#define LED_BEAM   9
#define LED_CLONE 10
#define LED1_DB1  11
#define LED2_DB1  12
#define LED_TAP   13
#define LED_INV   14
#define LED1_DB2  15
#define LED2_DB2  17

#define LED_ADKL  24
#define LED_ADKR  25

// BARGRAPHs

#define J1R1	 32 // Joystick 1 - Row 1 (Bargraph 1)
#define J1R2   33 // Joystick 1 - Row 2 (Bargraph 2)
#define J1R3   34 // Joystick 1 - Row 3 (Bargraph 3)
#define J1R4   35 // Joystick 1 - Row 4 (Bargraph 4)
#define J2R1	 36 // Joystick 2 - Row 1 (Bargraph 5)
#define J2R2   37 // Joystick 2 - Row 2 (Bargraph 6)
#define J2R3   38 // Joystick 2 - Row 3 (Bargraph 7)
#define J2R4   39 // Joystick 2 - Row 4 (Bargraph 8)

unsigned char GLOBAL_PROG_CHANGE;
unsigned char GLOBAL_PRESET_NO;				// N de preset en cours de jeu
unsigned char GLOBAL_PRESET_NAME[17]; // Nom du preset en cours de jeu
unsigned char GLOBAL_BROWSE_NO; 			// N de preset parcouru (en MODE_BANK)
unsigned char GLOBAL_BROWSE_NAME[17];	// Nom du preset parcouru (en MODE_BANK)

// Explications pour le BankStick
/*

J'ai 8*32kb = 256kb (car mes modules sont des 24LC256)
            = 256 x 1024 = 262144 octets (4 x 65536)

Plage d'addressage pour chaque module : 

		0x0000 (0)  0x7fff (32767)
	
Slection du module avant criture :
	
		MIOS_BANKSTICK_CtrlSet(0x07); // Slection du 8me BankStick

Lecture / Ecriture par octets :
		Un preset prend 363 octets (512 pour simplifier et laisser du FFU)
		32768/512 = 64
		Soit 64 presets stocks sur chaque module
		
		8 modules => 512 presets
		
		Le preset 0 est le preset par dfaut (celui qui sert de rfrence en cas d'initialisation voire 
		de factory reset).
		
		Les global settings (comportement pdale de program change)

Lecture / Ecriture par pages de 64 bytes :
		(Ca a l'air assez compliqu, la methode de lecture/criture par octets semble plus	simple)
		sur 32767 positions, a fait 512 pages pour 1 module,
		soit 4096 pages pour 8 modules

Donc je peux tout  fait dire que les 1024 premires pages sont rserves au presets
Je peux aussi dcider qu'un preset occupe 2 pages. 

Il faut dterminer de quelle place j'ai besoin pour un preset :
	
		Donnes globales :
	
		- Nom du preset (16 char)
		
		J'ai 8 axes de joysticks + 2 IR +  7 analog (2 softpots + 2 wheels + 2 knobs + 1 expression pedal)
		= 17 controleurs analogiques dont les paramtres sont  stocker
		Le switch pedal n'est pas considr comme controleur (il sert  changer de preset en mode voice 
		et inc/dec en mode edit)
		
		Par jeu de controleurs :
			
		- Nom du controleur			: 16 octets
		- NCC									:  1 octet
		- Canal MIDI de sortie 	:  1 octet (mme si c'est trop - c'est l'espace de base si je veux pas m'emmerder)
		- Sens du controleur 		:  1 octet (normal ou invers)
		
		> D-Beams :
		
		- Mode          				:  1 octet (Log/Lin)
		- RTZ (return to zero)	:  1 octet (Yes/No)

		> SoftPots :
		
		- Mode (Note on/off, Note on/off + CTL, CTL, CTL with return to zero)
		- Note min
		- Note max

		> bouton Freeze :
		
		- Mode (Note on off, CTL)
		
   16 caractres pour les noms des paramtres associs
   aux controleurs :
		
   12345678901234567890

   (mode VOICE)
  
L1 FILTER CUTOFF...:XXX			J1X ou J3X ou PITCH  ou POT_LEFT
L2 FILTER RESONANCE:XXX			J1Y ou J3Y ou MOD    ou POT_RIGHT
L3 LFO RATE........:XXX			J2X ou J4X ou DBEAM1 ou SOFTPOT1
L4 LFO DEPTH.......:XXX			J2Y ou J4Y ou DBEAM2 ou SOFTPOT2

*/

/////////////////////////////////////////////////////////////////////////////
//	This is an assembly optimized function which scales a 7bit value (0-127)
//	between a minimum and maximum value (eg 0 and 64)
//  Code by TK http://www.ucapps.de
/////////////////////////////////////////////////////////////////////////////

unsigned char ACMath_Scale(unsigned char v, unsigned char min, unsigned char max) __wparam
{
	// scales an 7-bit (0-255) value between min & max
#ifdef _DEBUG_C
	unsigned char PRODL;
	unsigned char PRODH;
	unsigned int  PRODLH;
	
	if(v == 0) { return 0; }
	PRODL = v << 1;			// 8bit value
	PRODH = max - min + 1;	// range
	PRODLH = PRODL * PRODH;
	return min + (unsigned char)(PRODLH >> 8);
#else
	if(v == 0) { return 0; }
	// scaled value is (<8-bit> * ) >> 8
	PRODL = v << 1;			// 8bit value
	PRODH = max - min + 1;	// range
	__asm
		movf _PRODL, W		// PRODL in W
		mulwf _PRODH, 0		// W * PRODH => result is 16bit { PRODH:PRODL }
	__endasm;
	
	return min + PRODH;		// return (min + (PRODH:PRODL / 16) );
#endif
}

// Affichage d'une valeur sur 3 caractres complte par des 0
void ADK_FILLED_0(unsigned char valeur)
{
	if (valeur <10)
		{ MIOS_LCD_PrintCString("00");MIOS_LCD_PrintBCD1(valeur); }
	else if (valeur <100)
		{ MIOS_LCD_PrintCString("0"); MIOS_LCD_PrintBCD2(valeur); }
	else
		{ MIOS_LCD_PrintBCD3(valeur); }
}

char IS_RIGHT(unsigned char pin)
{
	if (   pin==J2X_PIN || pin==J2Y_PIN || pin==J4X_PIN || pin==J4Y_PIN
		  || pin==DBEAM2_PIN || pin==POT_RIGHT || pin==MOD_PIN || pin==SOFTPOT2_PIN || pin==EXPPED_PIN)
		return -1;
	else
		return 0;
}

char IS_LEFT(unsigned char pin)
{
	if (   pin==J1X_PIN || pin==J1Y_PIN || pin==J3X_PIN || pin==J3Y_PIN
		  || pin==DBEAM1_PIN || pin==POT_LEFT || pin==PITCH_PIN || pin==SOFTPOT1_PIN || pin==EXPPED_PIN)
		return -1;
	else
		return 0;
}

unsigned char GET_INDEX_FROM_PIN(unsigned char pin)
{
	unsigned char index;

	if     (pin==J1X_PIN)      index=0; 
	else if(pin==J1Y_PIN)      index=1; 
	else if(pin==J2X_PIN)      index=2; 
	else if(pin==J2Y_PIN)      index=3; 
  else if(pin==J3X_PIN)      index=4; 
  else if(pin==J3Y_PIN)      index=5; 
  else if(pin==J4X_PIN)      index=6; 
  else if(pin==J4Y_PIN)      index=7; 
  else if(pin==PITCH_PIN)    index=8; 
  else if(pin==MOD_PIN)      index=9; 
  else if(pin==DBEAM1_PIN)   index=10;
  else if(pin==DBEAM2_PIN)   index=11;
  else if(pin==POT_LEFT)     index=12;
  else if(pin==POT_RIGHT)    index=13;
  else if(pin==SOFTPOT1_PIN) index=14;
  else if(pin==SOFTPOT2_PIN) index=15;
	else if(pin==EXPPED_PIN)   index=16;
		
	return index;
}

unsigned char GET_PIN_FROM_INDEX(unsigned char index)
{
	unsigned char pin;
	
	if      (index==0)  pin=J1X_PIN;
	else if (index==1)  pin=J1Y_PIN;
	else if (index==2)  pin=J2X_PIN;
	else if (index==3)  pin=J2Y_PIN;
	else if (index==4)  pin=J3X_PIN;
	else if (index==5)  pin=J3Y_PIN;
	else if (index==6)  pin=J4X_PIN;
	else if (index==7)  pin=J4Y_PIN;
	else if (index==8)  pin=PITCH_PIN;
	else if (index==9)  pin=MOD_PIN;
	else if (index==10) pin=DBEAM1_PIN;
	else if (index==11) pin=DBEAM2_PIN;
	else if (index==12) pin=POT_LEFT;
	else if (index==13) pin=POT_RIGHT;
	else if (index==14) pin=SOFTPOT1_PIN;
	else if (index==15) pin=SOFTPOT2_PIN;
	else if (index==16) pin=EXPPED_PIN;

	return pin;
}

unsigned char GET_POS_LINE_FROM_PIN(unsigned char pin)
{
	unsigned char line;

	if     (pin==J1X_PIN)      line=POS_LINE_1;
	else if(pin==J1Y_PIN)      line=POS_LINE_2;
	else if(pin==J2X_PIN)      line=POS_LINE_3;
	else if(pin==J2Y_PIN)      line=POS_LINE_4;
  else if(pin==J3X_PIN)      line=POS_LINE_1;
  else if(pin==J3Y_PIN)      line=POS_LINE_2;
  else if(pin==J4X_PIN)      line=POS_LINE_3;
  else if(pin==J4Y_PIN)      line=POS_LINE_4;
  else if(pin==PITCH_PIN)    line=POS_LINE_1;
  else if(pin==MOD_PIN)      line=POS_LINE_2;
  else if(pin==DBEAM1_PIN)   line=POS_LINE_3;
  else if(pin==DBEAM2_PIN)   line=POS_LINE_4;
  else if(pin==POT_LEFT)     line=POS_LINE_1;
  else if(pin==POT_RIGHT)    line=POS_LINE_2;
  else if(pin==SOFTPOT1_PIN) line=POS_LINE_3;
  else if(pin==SOFTPOT2_PIN) line=POS_LINE_4;
	else if(pin==EXPPED_PIN)   line=POS_LINE_4;
		
	return line;
}

unsigned char TRANSFORM_VALUE(unsigned char pin, unsigned char valeur7bits)
{
	unsigned char idx = GET_INDEX_FROM_PIN(pin);
	
	// Correction des sens des bargraphs par rapport aux soudures actuelles
	//if (  (SPECIAL_CALL!=1/* || BTN_FREEZE_PUSHED!=1*/)
	if (BTN_FREEZE_PUSHED!=1 && (pin==J2Y_PIN || pin==J3X_PIN || pin==POT_LEFT || pin==POT_RIGHT || pin==MOD_PIN))
	{
		valeur7bits = 127-valeur7bits;
		//valeur10bits = 1023 - valeur10bits;
	}
	
	// Je corrige le sens au besoin (sauf en cas d'appui sur le bouton Freeze/Cut 
	// o je transmets l'une des bornes (Min ou Max) et donc je ne souhaite pas inverser
	if (CTL_INV[idx]==ON && BTN_FREEZE_PUSHED!=1) valeur7bits = 127 - valeur7bits;

	// Je corrige Min/Max (sauf en edition o je veux entendre la plage totale d'utilisation)
	if (ADK_MODE!=MODE_EDIT)
	{
		valeur7bits = ACMath_Scale(valeur7bits, CTL_MIN[idx], CTL_MAX[idx]);
		// En mode VOICE, je veille au respect de la borne inf
		if (CTL_MIN[idx]!=0 && valeur7bits==0)
			valeur7bits = CTL_MIN[idx];
	}
	return valeur7bits;		
}

void DISPLAY_CTL(int pin, int value)
{
	unsigned char line_name, line_value;  

	// Les affichages de noms/valeurs ne concernent que l'cran VOICE et le premier cran d'dition
	if (ADK_MODE==MODE_VOICE || ADK_MODE==MODE_EDIT)
	{
		// Premier controleur touch depuis l'initialisation de l'cran
		// (qu'on soit en mode VOICE ou EDIT) -> J'efface l'cran pour construire le nouveau
		if (GLOBAL_LAST_TOUCHED_PIN == 255) MIOS_LCD_Clear();		
	
		// Sur quelle ligne dois-je faire l'affichage du nom et de la valeur du controleur ?
		if (ADK_MODE == MODE_EDIT) 
		{
			line_name = POS_LINE_1;
			line_value = POS_LINE_2;
		}
		else
		{
			line_name = GET_POS_LINE_FROM_PIN(pin); 
			line_value = line_name;
		}
		
		// Si c'est un controleur diffrent du dernier qui a parl
		if (GLOBAL_LAST_TOUCHED_PIN != pin)
		{
			GLOBAL_LAST_TOUCHED_PIN = pin;
			GLOBAL_LAST_TOUCHED_IDX = GET_INDEX_FROM_PIN(pin);
	
			if (ADK_MODE == MODE_VOICE)
			{
				// En mode Voice je prpare l'affichage du dernier qui a parl
				MIOS_LCD_CursorSet(POS_LINE_1+16);
				MIOS_LCD_PrintChar(':');
				MIOS_LCD_CursorSet(POS_LINE_2+16);
				MIOS_LCD_PrintChar(':');
				MIOS_LCD_CursorSet(POS_LINE_3+16);
				MIOS_LCD_PrintChar(':');
				MIOS_LCD_CursorSet(POS_LINE_4+16);
				MIOS_LCD_PrintChar(':');
			}
			else if (ADK_MODE == MODE_EDIT)
			{
				// J'affiche la flche sous le nom (edition du nom)
				MIOS_LCD_CursorSet(POS_LINE_2);
				MIOS_LCD_PrintCString("                ");
				MIOS_LCD_CursorSet(POS_LINE_2 + EDIT_POS);	
				MIOS_LCD_PrintChar(0);
	
				// J'affiche le numro de contrleur continu
				MIOS_LCD_CursorSet(POS_LINE_1+16);
				MIOS_LCD_PrintChar(':');
				MIOS_LCD_PrintBCD3(CTL_CC[GLOBAL_LAST_TOUCHED_IDX]);
	
				// J'affiche le canal MIDI utilis
				MIOS_LCD_CursorSet(POS_LINE_3);
				MIOS_LCD_PrintCString("Midi channel    :");
				MIOS_LCD_PrintBCD3(CTL_MIDICH[GLOBAL_LAST_TOUCHED_IDX]);
	
				// J'affiche les mins et max
				MIOS_LCD_CursorSet(POS_LINE_4);
				MIOS_LCD_PrintCString("Min");
				MIOS_LCD_PrintChar(2); // Flche vers la droite
				MIOS_LCD_PrintBCD3(CTL_MIN[GLOBAL_LAST_TOUCHED_IDX]);
				MIOS_LCD_PrintCString("      ");
				MIOS_LCD_PrintBCD3(CTL_MAX[GLOBAL_LAST_TOUCHED_IDX]);
				MIOS_LCD_PrintChar(4); // Flche vers la gauche
				MIOS_LCD_PrintCString("Max");
				
				// Si c'est un DBeam, j'affiche le mode (LIN/LOG) sinon les deux points :
				MIOS_LCD_CursorSet(POS_LINE_2+16);
				if (pin==DBEAM1_PIN || pin==DBEAM2_PIN)
				{
					if (pin==DBEAM1_PIN)
					{
						if (DB_LINLOG[0]==LIN) MIOS_LCD_PrintChar(6);
						else                   MIOS_LCD_PrintChar(7);
					}
					else
					{
						if (DB_LINLOG[1]==LIN) MIOS_LCD_PrintChar(6);
						else                   MIOS_LCD_PrintChar(7);
					}
				}
				else
					MIOS_LCD_PrintChar(':');
			}
			
			// J'affiche le nouveau nom
			MIOS_LCD_CursorSet(line_name);
			MIOS_LCD_PrintCString(CTL_NAME_PART1[GLOBAL_LAST_TOUCHED_IDX]);
			MIOS_LCD_PrintCString(CTL_NAME_PART2[GLOBAL_LAST_TOUCHED_IDX]);
	
			if (ADK_MODE == MODE_VOICE) 		
				MIOS_LCD_PrintChar(2);
			else
				MIOS_LCD_PrintChar(':');
		}
		
		// et pour finir j'affiche la valeur
		MIOS_LCD_CursorSet(line_value+17);
		MIOS_LCD_PrintBCD3(value);
	}
}

void JUST_PLAY()
{
	MIOS_LCD_Clear();
	MIOS_LCD_CursorSet(POS_LINE_1);	MIOS_LCD_PrintCString("--------------------");
	MIOS_LCD_CursorSet(POS_LINE_2);
	ADK_FILLED_0(GLOBAL_PRESET_NO);
	MIOS_LCD_PrintChar(':');
	MIOS_LCD_PrintCString(GLOBAL_PRESET_NAME);
	MIOS_LCD_CursorSet(POS_LINE_3+1);	MIOS_LCD_PrintChar(4);
	MIOS_LCD_PrintCString("Play the Arkade!");MIOS_LCD_PrintChar(2);
	MIOS_LCD_CursorSet(POS_LINE_4);	MIOS_LCD_PrintCString("--------------------");
}

// Rinitialisation des valeurs d'un contrleur aux valeurs par dfaut
void REINIT_CTL(int idx)
{
	/*
	int i;
	char def_name[16];

	if      (idx==0) def_name = "Joystick 1 X";
	else if (idx==1) def_name = "Joystick 1 Y";
	
	//CTL_NAME_PART1[0] = "AAAAAA\0";
	
	for (i=0; i<9; i++)
	{
		CTL_NAME_PART1[0][i] = def_name[i];
	}
	*/
}

unsigned char GET_BANKSTICK(int presetno)
{
	// Presets 001  064 : bankstick : 1 / memoffset : (presetno-1)*512
	// Presets 065  128 : bankstick : 2 / memoffset : (presetno-65)*512
	unsigned char bankstick;
	
	if (presetno < 65)
	{
		if (presetno == -1) bankstick = 0;
		else 								bankstick = 1;
	}
	else
		bankstick = 2;
		
	return bankstick;
}

int GET_MEM_OFFSET(int presetno)
{
	// Presets 001  064 : bankstick : 1 / memoffset : (presetno-1)*512
	// Presets 065  128 : bankstick : 2 / memoffset : (presetno-65)*512
	int mem_offset;

	if (presetno < 65)
	{
		if (presetno == -1) mem_offset = 0;
		else 								mem_offset = (presetno-1)<<9;
	}
	else
		mem_offset = (presetno-65)<<9;
		
	return mem_offset;
}

void LOAD_PRESET(int presetno)
{
	// Organisation du bankstick : 
	// Le module 0 stocke le preset par dfaut ainsi que les paramtres globaux de l'Arkade
	// Les modules 1 et 2 stockent les presets
	// Chaque preset occupe 512 octets, chaque module fait 32 Ko (32768/512=64 presets)
	// Total=128 presets

	// Pour charger le preset par dfaut, 'presetno' doit valoir -1
	// Sinon les presets commencent  1 donc logiquement presetno ne vaut jamais zro
	
	// Slection du bankstick
	unsigned char bankstick = GET_BANKSTICK(presetno); 
	
	// Slection de l'offset mmoire
	int mem_offset = GET_MEM_OFFSET(presetno);

	unsigned char cpt=0;
	unsigned char ctl=0;
	unsigned char octet;

	if (presetno==0) return; // L'appel avec cette valeur ne doit pas se produire

	MEM_BUSY=1; // On flague pour que l'activit des controleurs ne perturbe pas l'I/O

	MIOS_BANKSTICK_CtrlSet(bankstick); // On slectionne le BankStick qui va bien
	
	// 1) Chargement des noms des controleurs	
	for (ctl=0; ctl<17; ctl++)
	{
		for (cpt=0; cpt<16; cpt++)
		{
			octet = MIOS_BANKSTICK_Read(mem_offset+(ctl << 4)+cpt); // << 4 = *16
			if (cpt<8)
				CTL_NAME_PART1[ctl][cpt] = octet;
			else
				CTL_NAME_PART2[ctl][cpt-8] = octet;
		}
		CTL_NAME_PART1[ctl][8] = '\0'; // On insre les bons terminateurs
		CTL_NAME_PART2[ctl][8] = '\0'; // de string sur les fins des 2 tableaux
	}
	
	// 2) CTL_LAST_VALUE[] :
	for (ctl=0; ctl<17; ctl++)
	{
		int pin = GET_PIN_FROM_INDEX(ctl);
		// Correction des sens des bargraphs par rapport aux soudures actuelles
		if (pin==J2Y_PIN || pin==J3X_PIN || pin==POT_LEFT || pin==POT_RIGHT || pin==MOD_PIN)
			CTL_LAST_VALUE[ctl] = 127-MIOS_AIN_Pin7bitGet(pin);
		else
			CTL_LAST_VALUE[ctl] = MIOS_AIN_Pin7bitGet(pin);
	}
	// Contrleurs dont l'initialisation doit tre force
	CTL_LAST_VALUE[GET_INDEX_FROM_PIN(SOFTPOT1_PIN)]=0;
	CTL_LAST_VALUE[GET_INDEX_FROM_PIN(SOFTPOT2_PIN)]=0;
	CTL_LAST_VALUE[GET_INDEX_FROM_PIN(DBEAM1_PIN)]=0;
	CTL_LAST_VALUE[GET_INDEX_FROM_PIN(DBEAM2_PIN)]=0;
	
	// 3) CTL_CC[] :
	for (ctl=0; ctl<17; ctl++)
	{
		CTL_CC[ctl] = MIOS_BANKSTICK_Read(mem_offset+272+ctl);
	}

	// 4) CTL_MIDICH[] :
	for (ctl=0; ctl<17; ctl++)
	{
		CTL_MIDICH[ctl] = MIOS_BANKSTICK_Read(mem_offset+289+ctl);
	}
	
	// 5) CTL_INV[] :
	for (ctl=0; ctl<17; ctl++)
	{
		CTL_INV[ctl] = MIOS_BANKSTICK_Read(mem_offset+306+ctl);
	}
	
	// 6) CTL_MIN[] et CTL_MAX[] :
	for (ctl=0; ctl<17; ctl++)
	{
		CTL_MIN[ctl] = MIOS_BANKSTICK_Read(mem_offset+323+ctl);
		CTL_MAX[ctl] = MIOS_BANKSTICK_Read(mem_offset+340+ctl);
	}
	
	// 7) LIN/LOG (DBeams)
	DB_LINLOG[0] = MIOS_BANKSTICK_Read(mem_offset+357);
	DB_LINLOG[1] = MIOS_BANKSTICK_Read(mem_offset+358);
	
	SP_MODE[0] = MIOS_BANKSTICK_Read(mem_offset+359);
	SP_RELEASE[0] = MIOS_BANKSTICK_Read(mem_offset+360);
	SP_MODE[1] = MIOS_BANKSTICK_Read(mem_offset+361);
	SP_RELEASE[1] = MIOS_BANKSTICK_Read(mem_offset+362);

	// 8) Nom du preset	
	for (cpt=0; cpt<16; cpt++)
	{
		GLOBAL_PRESET_NAME[cpt] = MIOS_BANKSTICK_Read(mem_offset+363+cpt);
	}
	GLOBAL_PRESET_NAME[16]='\0';
	
	MEM_BUSY=0;
}

void SAVE_PRESET()
{

}

void PRINT_PRESET_NAME(int presetno, int update)
{
	// Update permet de dfinir si on doit mettre  jour le tableau global
	// GLOBAL_BROWSE_NAME[]
	
	char name[17];
	unsigned char cpt;

	// Slection du bankstick
	unsigned char bankstick = GET_BANKSTICK(presetno); 
	
	// Offset mmoire sur le bankstick
	int mem_offset = GET_MEM_OFFSET(presetno);

	MEM_BUSY=1;
	MIOS_BANKSTICK_CtrlSet(bankstick); // On slectionne le BankStick qui va bien
	
	// Chargement du nom
	for (cpt=0; cpt<16; cpt++) 
	{
		name[cpt] = MIOS_BANKSTICK_Read(mem_offset+363+cpt);
		if (update==1) GLOBAL_BROWSE_NAME[cpt] = name[cpt];
	}
	name[16]='\0';
	if (update==1) GLOBAL_BROWSE_NAME[16]='\0';
	
	MIOS_LCD_PrintCString(name);
	
	MEM_BUSY=0;
}

void PRINT_DOTS_BANK(unsigned char preset)
{
	if (preset==GLOBAL_PRESET_NO) // celui en cours d'dition
		MIOS_LCD_PrintChar(1);
	else
		MIOS_LCD_PrintChar(':');
}

void SCREEN_BANK()
{
	char temp[17];
	
	MIOS_LCD_Clear();
	MIOS_LCD_CursorSet(POS_LINE_1);
	ADK_FILLED_0(GLOBAL_BROWSE_NO);
	PRINT_DOTS_BANK(GLOBAL_BROWSE_NO);
	PRINT_PRESET_NAME(GLOBAL_BROWSE_NO,1);

	MIOS_LCD_CursorSet(POS_LINE_2+4+EDIT_POS);
	MIOS_LCD_PrintChar(0);
	
	if (GLOBAL_BROWSE_NO<128)
	{
		MIOS_LCD_CursorSet(POS_LINE_3);
		ADK_FILLED_0(GLOBAL_BROWSE_NO+1);
		PRINT_DOTS_BANK(GLOBAL_BROWSE_NO+1);
		PRINT_PRESET_NAME(GLOBAL_BROWSE_NO+1,0);
	}
	if (GLOBAL_BROWSE_NO<127)
	{
		MIOS_LCD_CursorSet(POS_LINE_4);
		ADK_FILLED_0(GLOBAL_BROWSE_NO+2);
		PRINT_DOTS_BANK(GLOBAL_BROWSE_NO+2);
		PRINT_PRESET_NAME(GLOBAL_BROWSE_NO+2,0);
	}
}

void NOTE_ON(int pos)
{
  MIOS_MIDI_TxBufferPut(0x90); // Note Event at channel #1
  MIOS_MIDI_TxBufferPut(127-pos/*48*/);  // Note number (0..127)
  MIOS_MIDI_TxBufferPut(0x7f); // Velocity = (0..127)
} 
  
void ALL_NOTES_OFF()
{
	unsigned char i=0;
	for(i=0;i<15;i++)
	{
		MIOS_MIDI_TxBufferPut(176 + i);//Controller Chn i
		MIOS_MIDI_TxBufferPut(123);//ALL NOTE OFF 
		MIOS_MIDI_TxBufferPut(0x00);
	}
	return;
}

void ADK_ProgCh_INC()
{
	GLOBAL_PROG_CHANGE++;
	if (GLOBAL_PROG_CHANGE>128) GLOBAL_PROG_CHANGE=128;
	else
	{
		// Envoi du program change MIDI
		MIOS_MIDI_TxBufferPut(0xc0); // program change, channel #1
		MIOS_MIDI_TxBufferPut(GLOBAL_PROG_CHANGE-1); // program number
	}
	MIOS_LCD_CursorSet(POS_LINE_1);
	ADK_FILLED_0(GLOBAL_PROG_CHANGE);
}

void ADK_ProgCh_DEC()
{
	GLOBAL_PROG_CHANGE--;
	if (GLOBAL_PROG_CHANGE<1) GLOBAL_PROG_CHANGE=1;
	else
	{
		// Envoi du program change MIDI
		MIOS_MIDI_TxBufferPut(0xc0); // program change, channel #1
		MIOS_MIDI_TxBufferPut(GLOBAL_PROG_CHANGE-1); // program number
	}

	MIOS_LCD_CursorSet(POS_LINE_1);
	ADK_FILLED_0(GLOBAL_PROG_CHANGE);
}
