// CMM Config For MIDIBox Mixer
//      will create a Standard MIDI file (format 0) to
//      configure a MIDIBox Mixer to user preferences.

//      ANSI C, should br portable to all systems that support a command line.

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

int ShowUsage(char *command);
int StartFile(FILE *fp);
int EndFile(FILE *fp, int l);
int ParseChannel(char *opt, char *val, int chan, FILE *fp);
int mbyte(char *str);
int addevent(FILE *fp,unsigned char time,unsigned char status,unsigned char cc,unsigned char value);


int main(int argc, char **argv)
{
    FILE *mid;
    int channel;
    int x,c,len;

    if(argc < 5)
    {
        ShowUsage(argv[0]);
        return(1);
    }

    if(mid=fopen(argv[1],"w"))
    {
        if(0==strcmp(argv[2],"all"))
        {
           channel=16;
           printf("Set all channels\n");
        }
        else
        {
            channel=atoi(argv[2])-1;
            printf("Set channel %d\n",channel+1);
        }

        len=StartFile(mid);     // open output file, write header..

        for(x=3; x<(argc-1); x+=2)      // scan the command line by twos
        {
            if(0==strcmp(argv[x],"-save"))
            {
                len += ParseChannel(argv[x],argv[x+1],0,mid);
            }
            if(0==strcmp(argv[x],"-load"))
            {
                len += ParseChannel(argv[x],argv[x+1],0,mid);
            }
            else if(0==strcmp(argv[x],"-master"))
            {
                len += ParseChannel(argv[x],argv[x+1],0,mid);
            }
            else if(0==strcmp(argv[x],"-reset"))
            {
                len += ParseChannel(argv[x],argv[x+1],0,mid);
            }
            else if(channel<16)      // handle for one channel
            {
                len += ParseChannel(argv[x],argv[x+1],channel,mid);
            }
            else
            {                   // or loop 16 times for all channels
                for(c=0; c<16; c++)
                {
                    len += ParseChannel(argv[x],argv[x+1],c,mid);
                }
            }
        }

        EndFile(mid,len);       // complete the MIDI file
        fclose(mid);
    }
    else
    {
        printf("Cannot open file %s for writing\n",argv[1]);
    }
    return(0);
}

int ShowUsage(char *command)
{
    printf("Usage: %s filename chan option option...\n",command);
    printf("where filename is the name of the .mid file to create.\n");
    printf("where chan may be 1-16 or all\n");
    printf("Option List:\n");
    printf("-vol [volume level]\n");
    printf("\tvolume level may be 0 to 127\n");
    printf("-bal [balance]\n");
    printf("\tbalance may be 1(full left) to 64(center) to 127(full right)\n");
    printf("-pan [pan]\n");
    printf("\tpan may be 1(full left) to 64(center) to 127(full right)\n");
    printf("-exp [expression level]\n");
    printf("\texpression level may be 0 to 127\n");
    printf("-fx1 [effects 1 level]\n");
    printf("\teffects 1 level may be 0 to 127\n");
    printf("-fx2 [effects 2 level]\n");
    printf("\teffects 2 level may be 0 to 127\n");
    printf("-fx1p [effects 1 position]\n");
    printf("\teffects 1 position may be pre or post (before or after main volume fader)\n");
    printf("-fx2p [effects 2 position]\n");
    printf("\teffects 2 position may be pre or post (before or after main volume fader)\n");
    printf("-fx1m [effects 1 muted]\n");
    printf("\teffects 1 mute may be true or false\n");
    printf("-fx2m [effects 2 muted]\n");
    printf("\teffects 2 mute may be true or false\n");
    printf("-withfx [this channel supports FX sends?]\n");
    printf("\twithfx may be true or false\n");
    printf("-master [master volume level]\n");
    printf("\tmaster volume level may be 0 to 127\n");
    printf("-setflags [flags]\n");
    printf("\twhere flags may be 0 to 63\n");
    printf("-sendmix 0\n");
    printf("\tMBMixer will send current settings out MIDI port\n");    
    printf("-reset 0\n");
    printf("\twill reset the entire board to power-up condition (Load Bank 0)\n");
    printf("-load [bank]\n");
    printf("\tbank may be 0 to 63\n");
    printf("-save [bank]\n");
    printf("\tbank may be 0 to 63\n");
    printf("-save option, if used, should always be the last option in the line\n");
    return(0);
}

// write the header for our MIDI file
int StartFile(FILE *fp)
{       // this header is format 0, 1 track, 96 ppq
    char header[]={'M','T','h','d',0,0,0,6,0,0,0,1,0,96,'M','T','r','k',0,0,0,0};
    int x;
    for(x=0;x<22;x++)
    {
        fputc(header[x],fp);
    }
    return(0);         // return the length (so far) of the new track
}

int EndFile(FILE *fp, int l)
{
    unsigned char LenH,Len2,Len1,LenL;
    unsigned long length;
    
    length = l+4;               // add length of "End Of Track" event
    LenL=length&0xFF;           // convert length to 4 bytes
    Len1=(length>>8)&0xFF;
    Len2=(length>>16)&0xFF;
    LenH=(length>>24)&0xFF;

    fputc(0x00,fp);             // time delay zero
    fputc(0xFF,fp);             // write "End of Track" event
    fputc(0x2F,fp);
    fputc(0x00,fp);
    fseek(fp,0x12,SEEK_SET);    // go back to track length
    fputc(LenH,fp);             // overwrite correct track length
    fputc(Len2,fp);
    fputc(Len1,fp);
    fputc(LenL,fp);
	return(0);
}

// atoi, limited to 0-127 values.
int mbyte(char *str)
{
    int x;
    x=atoi(str);
    if(x<0)
    {
        x=0;
    }
    if(x>127)
    {
        x=127;
    }
    return(x);
}

int addevent(FILE *fp,unsigned char time,unsigned char status,unsigned char cc,unsigned char value)
{
    fputc(time,fp);
    fputc(status,fp);
    fputc(cc,fp);
    fputc(value,fp);
    return(0);
}

// read a pair of commandline tokens, and try to figure them out.
// write MIDI commands to the file, return number of bytes written
int ParseChannel(char *opt, char *val, int chan, FILE *fp)
{
    unsigned char status = 0xB0;
    
    if(0==strcmp(opt,"-vol"))
    {
        addevent(fp,0,status|chan,7,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-bal"))
    {
        addevent(fp,0,status|chan,8,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-pan"))
    {
        addevent(fp,0,status|chan,10,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-exp"))
    {
        addevent(fp,0,status|chan,11,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-fx1"))
    {
        addevent(fp,0,status|chan,12,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-fx2"))
    {
        addevent(fp,0,status|chan,13,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-fx1p"))
    {
        if(0==strcmp(val,"pre"))
        {
            addevent(fp,0,status|chan,80,127);
            return(4);
        }
        else if(0==strcmp(val,"post"))
        {
            addevent(fp,0,status|chan,80,0);
            return(4);
        }
        else
        {
            printf("Cannot understand setting %s for -fx1p, skipping this option\n",val);
            printf("Try using 'pre' or 'post' for -fx1p settings\n");
            return(0);
        }
    }
    else if(0==strcmp(opt,"-fx2p"))
    {
        if(0==strcmp(val,"pre"))
        {
            addevent(fp,0,status|chan,81,127);
            return(4);
        }
        else if(0==strcmp(val,"post"))
        {
            addevent(fp,0,status|chan,81,0);
            return(4);
        }
        else
        {
            printf("Cannot understand setting %s for -fx2p, skipping this option\n",val);
            printf("Try using 'pre' or 'post' for -fx2p settings\n");
            return(0);
        }
    }
    else if(0==strcmp(opt,"-fx1m"))
    {
        if(0==strcmp(val,"true"))
        {
            addevent(fp,0,status|chan,82,127);
            return(4);
        }
        else if(0==strcmp(val,"false"))
        {
            addevent(fp,0,status|chan,82,0);
            return(4);
        }
        else
        {
            printf("Cannot understand setting %s for -fx1m, skipping this option\n",val);
            printf("Try using 'true' or 'false' for -fx1m settings\n");
            return(0);
        }
    }
    else if(0==strcmp(opt,"-fx2m"))
    {
        if(0==strcmp(val,"true"))
        {
            addevent(fp,0,status|chan,83,127);
            return(4);
        }
        else if(0==strcmp(val,"false"))
        {
            addevent(fp,0,status|chan,83,0);
            return(4);
        }
        else
        {
            printf("Cannot understand setting %s for -fx2m, skipping this option\n",val);
            printf("Try using 'true' or 'false' for -fx2m settings\n");
            return(0);
        }
    }
    else if(0==strcmp(opt,"-withfx"))
    {
        if(0==strcmp(val,"true"))
        {
            addevent(fp,0,status|chan,15,127);
            return(4);
        }
        else if(0==strcmp(val,"false"))
        {
            addevent(fp,0,status|chan,15,0);
            return(4);
        }
        else
        {
            printf("Cannot understand setting %s for -withfx, skipping this option\n",val);
            printf("Try using 'true' or 'false' for -withfx settings\n");
            return(0);
        }
    }
    else if(0==strcmp(opt,"-setflags"))
    {
        addevent(fp,0,status|chan,20,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-sendmix"))
    {
        addevent(fp,0,status|chan,21,0);
        return(4);
    }
    else if(0==strcmp(opt,"-master"))
    {
        addevent(fp,0,status|chan,23,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-reset"))
    {
        addevent(fp,0,status|chan,120,0);
        return(4);
    }

    else if(0==strcmp(opt,"-d1"))
    {
        fprintf(fp,"%c%c%c%c%c%c%c%c%c%s%c",0x00,0xF0,0,0,0x7E,0x40,0,0x08,0,val,0xF7);
        return(10+strlen(val));
    }
    else if(0==strcmp(opt,"-d2"))
    {
        fprintf(fp,"%c%c%c%c%c%c%c%c%c%s%c",0x00,0xF0,0,0,0x7E,0x80,0,0x08,0,val,0xF7);
        return(10+strlen(val));
    }

    else if(0==strcmp(opt,"-load"))
    {
        addevent(fp,0,status|chan,24,mbyte(val));
        return(4);
    }
    else if(0==strcmp(opt,"-save"))
    {
        addevent(fp,0,status|chan,22,mbyte(val));
        return(4);
    }
    else
    {
        printf("cannot understand option %s, skipping this option\n",opt);
        return(0);
    }
}
